<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invoice;
use App\Models\Product; // همچنان به این مدل نیاز داریم
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CancelPendingInvoices extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'invoices:cancel-pending';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Cancel invoices that have been pending for more than 24 hours and restore stock'; // ✅ توضیحات به‌روز شد

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting to cancel expired pending invoices...');
        
        // ۱. پیدا کردن فاکتورهای منقضی شده
        // (و لود کردن محصولات مرتبط با آن‌ها از جدول واسط)
        $expiredInvoices = Invoice::where('status', 'pending')
                                ->where('created_at', '<=', now()->subHours(24))
                                ->with('products') // ✅ مهم: لود کردن رابطه چند-به-چند محصولات
                                ->get();

        if ($expiredInvoices->isEmpty()) {
            $this->info('No expired invoices found.');
            return 0;
        }

        $this->info($expiredInvoices->count() . ' expired invoices found. Processing...');

        foreach ($expiredInvoices as $invoice) {
            try {
                DB::transaction(function () use ($invoice) {
                    
                    // ✅ ۲. بازگرداندن موجودی انبار (منطق اصلاح شده)
                    // (این حلقه تمام محصولات داخل فاکتور، چه تکی و چه باندل، را پردازش می‌کند)
                    foreach ($invoice->products as $product) {
                        
                        // $product->pivot->quantity تعداد دقیق ثبت شده در فاکتور است
                        $quantityToRestore = $product->pivot->quantity;

                        // برای جلوگیری از Race Condition، ردیف محصول را قفل می‌کنیم
                        $productToUpdate = Product::lockForUpdate()->find($product->id);
                        
                        if ($productToUpdate) {
                            // موجودی را به انبار باز می‌گردانیم
                            $productToUpdate->increment('stock', $quantityToRestore);
                            Log::info("Invoice #{$invoice->id}: Restored {$quantityToRestore} stock for Product #{$productToUpdate->id} ('{$productToUpdate->name}').");
                        } else {
                            // اگر محصول در این فاصله حذف شده بود
                            Log::warning("Invoice #{$invoice->id}: Product #{$product->id} not found for stock restoration.");
                        }
                    }
                    
                    // ۳. تغییر وضعیت فاکتور به 'rejected' (لغو شده)
                    $invoice->update(['status' => 'rejected']);
                });
                
                $this->info('Invoice #' . $invoice->id . ' has been canceled and stock restored for all items.');
                Log::info('Invoice #' . $invoice->id . ' has been canceled and stock restored for all items.');

            } catch (\Exception $e) {
                $this->error('Failed to cancel invoice #' . $invoice->id . ': ' . $e->getMessage());
                Log::error('Failed to cancel invoice #' . $invoice->id . ': ' . $e->getMessage());
            }
        }

        $this->info('All expired invoices processed.');
        return 0;
    }
}