<?php

namespace App\Http\Controllers\Accountant;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule; 

class InvoiceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // ... (کد index بدون تغییر) ...
        $query = Invoice::query()->with(['user', 'customer'])
                                ->withCount('products');
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function($q) use ($searchTerm) {
                $q->where('id', 'like', "%{$searchTerm}%")
                  ->orWhereHas('customer', function($subq) use ($searchTerm) {
                      $subq->where('name', 'like', "%{$searchTerm}%");
                  })
                  ->orWhereHas('user', function($subq) use ($searchTerm) { 
                      $subq->where('name', 'like', "%{$searchTerm}%");
                  });
            });
        }
        $invoices = $query->latest()->paginate(15)->appends($request->query());
        return view('regional-manager.invoices.index', [
            'invoices' => $invoices,
            'is_accountant' => true 
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Invoice $invoice)
    {
        $invoice->load(['user', 'customer', 'planOption.plan', 'products']);
        $documents = json_decode($invoice->document_paths, true);
        
        // ✅ START: خواندن فایل‌های چک از دیتابیس
        $cheque_documents = json_decode($invoice->cheque_document_paths, true);
        // ✅ END: خواندن فایل‌های چک

        // (استفاده از همان ویو مدیر منطقه)
        return view('regional-manager.invoices.show', [
            'invoice' => $invoice,
            'documents' => $documents,
            'cheque_documents' => $cheque_documents, // ✅ پاس دادن فایل‌های چک به ویو
            'is_accountant' => true 
        ]);
    }
    
    
    public function updateStatus(Request $request, Invoice $invoice)
    {
        // ... (کد updateStatus که در فاز قبل اصلاح شد، بدون تغییر) ...
        if (auth()->user()->role !== 'accountant' && auth()->user()->role !== 'super_admin') {
            abort(403);
        }
        $request->validate([
            'status' => ['required', 'string', Rule::in([
                'pending', 'processing', 'ready_to_ship', 'shipped', 'delivered', 'returned', 'rejected'
            ])]
        ]);
        $invoice->update(['status' => $request->status]);
        return back()->with('success', 'وضعیت پیش فاکتور با موفقیت به‌روز شد.');
    }


    public function printInvoice(Invoice $invoice)
        {
            $invoice->load('customer', 'planOption.plan', 'products');
            $settings = Setting::pluck('value', 'key')->all();
            
            // ✅✅✅ تغییر اینجاست
            return view('agent.show-invoice', [ 
                'invoice' => $invoice,
                'settings' => $settings
            ]);
        }
}