<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bundle;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class BundleController extends Controller
{


    public function index()
    {
        $bundles = Bundle::withCount('products')->latest()->paginate(15);
        return view('admin.bundles.index', compact('bundles'));
    }

    public function create()
    {
        $products = Product::where('status', 'active')->select('id', 'name', 'code', 'stock')->get();
        return view('admin.bundles.create', compact('products'));
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:bundles'],
            'price' => ['required', 'numeric', 'min:0'],
            'status' => ['required', Rule::in(['1', '0'])],
            'products' => ['required', 'array', 'min:1'],
            'products.*.id' => ['required', 'exists:products,id'],
            'products.*.quantity' => ['required', 'integer', 'min:1'],
        ]);

        try {
            DB::beginTransaction();
            
            $bundle = Bundle::create([
                'name' => $validatedData['name'],
                'price' => $validatedData['price'],
                'status' => $validatedData['status'],
            ]);

            // آماده‌سازی داده‌های جدول واسط
            $productsToSync = [];
            foreach ($validatedData['products'] as $product) {
                // (جلوگیری از ثبت محصول تکراری در فرم)
                if (isset($productsToSync[$product['id']])) {
                    $productsToSync[$product['id']]['quantity'] += $product['quantity'];
                } else {
                    $productsToSync[$product['id']] = ['quantity' => $product['quantity']];
                }
            }
            
            $bundle->products()->sync($productsToSync);

            DB::commit();
            return redirect()->route('bundles.index')->with('success', 'باندل با موفقیت ایجاد شد.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'خطا در ایجاد باندل: ' . $e->getMessage());
        }
    }

    public function edit(Bundle $bundle)
    {
        $bundle->load('products'); // بارگذاری محصولات باندل
        $products = Product::where('status', 'active')->select('id', 'name', 'code', 'stock')->get();
        
        // آماده‌سازی محصولات باندل برای نمایش در فرم Alpine.js
        $selectedProducts = $bundle->products->map(function($p) {
            return [
                'id' => $p->id,
                'name' => $p->name . ' (' . $p->code . ')',
                'quantity' => $p->pivot->quantity
            ];
        });
        
        return view('admin.bundles.edit', compact('bundle', 'products', 'selectedProducts'));
    }

    public function update(Request $request, Bundle $bundle)
    {
        $validatedData = $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('bundles')->ignore($bundle->id)],
            'price' => ['required', 'numeric', 'min:0'],
            'status' => ['required', Rule::in(['1', '0'])],
            'products' => ['required', 'array', 'min:1'],
            'products.*.id' => ['required', 'exists:products,id'],
            'products.*.quantity' => ['required', 'integer', 'min:1'],
        ]);

        try {
            DB::beginTransaction();
            
            $bundle->update([
                'name' => $validatedData['name'],
                'price' => $validatedData['price'],
                'status' => $validatedData['status'],
            ]);

            // آماده‌سازی داده‌های جدول واسط
            $productsToSync = [];
            foreach ($validatedData['products'] as $product) {
                if (isset($productsToSync[$product['id']])) {
                    $productsToSync[$product['id']]['quantity'] += $product['quantity'];
                } else {
                    $productsToSync[$product['id']] = ['quantity' => $product['quantity']];
                }
            }
            
            $bundle->products()->sync($productsToSync);

            DB::commit();
            return redirect()->route('bundles.index')->with('success', 'باندل با موفقیت به‌روزرسانی شد.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'خطا در به‌روزرسانی باندل: ' . $e->getMessage());
        }
    }

    public function destroy(Bundle $bundle)
    {
        // (جدول واسط به صورت خودکار با حذف باندل پاک می‌شود)
        $bundle->delete();
        return back()->with('success', 'باندل با موفقیت حذف شد.');
    }
}