<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;

class InventoryController extends Controller
{
    /**
     * نمایش لیست محصولات برای مدیریت انبار
     */
    public function index(Request $request)
    {
        // فقط سوپر ادمین به این بخش دسترسی دارد
        if (auth()->user()->role !== 'super_admin') abort(403);

        $query = Product::query()->select('id', 'name', 'code', 'stock');

        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('code', 'like', '%' . $request->search . '%');
        }

        $products = $query->latest()->paginate(25)->appends($request->query());

        return view('admin.inventory.index', compact('products'));
    }

    /**
     * آپدیت کردن موجودی یک محصول
     */
    public function updateStock(Request $request, Product $product)
    {
        // فقط سوپر ادمین به این بخش دسترسی دارد
        if (auth()->user()->role !== 'super_admin') abort(403);
        
        // اعتبارسنجی ساده برای تنظیم موجودی
        $validated = $request->validate([
            'stock_action' => ['required', Rule::in(['set', 'add', 'subtract'])],
            'quantity' => ['required', 'integer', 'min:0'],
        ]);
        
        try {
            if ($validated['stock_action'] === 'set') {
                // تنظیم مستقیم
                $product->stock = $validated['quantity'];
            } elseif ($validated['stock_action'] === 'add') {
                // اضافه کردن
                $product->stock += $validated['quantity'];
            } elseif ($validated['stock_action'] === 'subtract') {
                // کم کردن
                if ($product->stock < $validated['quantity']) {
                    throw new \Exception('موجودی انبار کافی نیست.');
                }
                $product->stock -= $validated['quantity'];
            }
            
            $product->save();
            
            return back()->with('success', 'موجودی ' . $product->name . ' با موفقیت به‌روز شد.');

        } catch (\Exception $e) {
            return back()->with('error', 'خطا: ' . $e->getMessage());
        }
    }
}