<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\PlanOption; // ✅ این را برای واکشی گزینه‌ها اضافه می‌کنیم
use App\Models\Category; 
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage; 
use Illuminate\Support\Facades\DB; 
use App\Imports\ProductsImport;
use Maatwebsite\Excel\Facades\Excel;
use Maatwebsite\Excel\Validators\ValidationException;

class ProductController extends Controller
{
    public function __construct()
    {
        // (این Policyها از قبل به درستی کار می‌کردند)
        $this->authorizeResource(Product::class, 'product');
    }

    public function showImportForm()
    {
        $this->authorize('create', Product::class);
        return view('admin.products.import');
    }

    public function import(Request $request)
    {
        // ... (این متد بدون تغییر باقی می‌ماند) ...
        $this->authorize('create', Product::class);
        $request->validate([
            'file' => 'required|mimes:xlsx,csv'
        ]);

        try {
            Excel::import(new ProductsImport, $request->file('file'));
        } catch (ValidationException $e) {
             $failures = $e->failures();
             $errorMessages = [];
             foreach ($failures as $failure) {
                 $errorMessages[] = 'خطا در ردیف ' . $failure->row() . ': ' . implode(', ', $failure->errors());
             }
             return back()->with('error', implode('<br>', $errorMessages));
        }
        
        return redirect()->route('admin.products.index')->with('success', 'محصولات با موفقیت درون‌ریزی شدند.');
    }

    public function index(Request $request)
    {
        // ... (این متد بدون تغییر باقی می‌ماند) ...
        $query = Product::query();

        if ($request->filled('status')) {
            if ($request->status == 'in_stock') {
                $query->where('stock', '>', 0);
            } elseif ($request->status == 'out_of_stock') {
                $query->where('stock', '=', 0);
            }
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                ->orWhere('code', 'like', '%' . $request->search . '%');
            });
        }

        $products = $query->with('category')->latest()->paginate(12);

        return view('admin.products.index', compact('products'));
    }

    public function create()
    {
        // $plans = Plan::all(); // ❌ حذف شد
        
        // ✅✅✅ جایگزین شد: ما گزینه‌ها را به همراه نام طرح والدشان واکشی می‌کنیم
        $planOptions = PlanOption::with('plan')->get(); 
        
        $categories = Category::all(); 
        
        // ✅ متغیر به ویو پاس داده شده عوض شد
        return view('admin.products.create', compact('planOptions', 'categories')); 
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|unique:products|max:255',
            'category_id' => 'nullable|exists:categories,id', 
            'price' => 'required|numeric|min:0', 
            'discount_price' => 'nullable|numeric|min:0|lt:price', 
            'stock' => 'required|integer',
            'status' => 'required|in:active,inactive',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            
            // ✅✅✅ این بخش تغییر کرد ✅✅✅
            'plan_options' => 'nullable|array', // نام فیلد عوض شد
            'plan_options.*' => 'exists:plan_options,id', // اعتبار سنجی گزینه‌ها
            
            // ✅✅✅ START: اعتبارسنجی ویژگی‌ها ✅✅✅
            'features' => 'nullable|array|max:10', // حداکثر 10 ویژگی
            'features.*.key' => 'required_with:features.*.value|string|max:255',
            'features.*.value' => 'required_with:features.*.key|string|max:255',
            // ✅✅✅ END: اعتبارسنجی ویژگی‌ها ✅✅✅
        ]);

        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('products', 'public');
            $validatedData['image_path'] = $path;
        }
        
        $validatedData['discount_price'] = $validatedData['discount_price'] ?? null;
        
        // ✅✅✅ START: آماده‌سازی ویژگی‌ها برای ذخیره ✅✅✅
        // (فیلتر کردن ردیف‌های خالی احتمالی)
        $validatedData['features'] = $this->prepareFeatures($request->features);
        // ✅✅✅ END: آماده‌سازی ویژگی‌ها ✅✅✅

        $product = Product::create($validatedData);
        
        // ✅✅✅ این بخش تغییر کرد ✅✅✅
        $product->planOptions()->sync($request->plan_options); // اتصال به گزینه‌ها

        return redirect()->route('admin.products.index');
    }

    public function edit(Product $product)
    {
        // $plans = Plan::all(); // ❌ حذف شد
        
        // ✅✅✅ جایگزین شد ✅✅✅
        $planOptions = PlanOption::with('plan')->get(); 
        $categories = Category::all(); 
        
        // ✅ بارگذاری گزینه‌های متصل شده فعلی
        $product->load('planOptions'); 
        
        // ✅ متغیرها عوض شدند
        return view('admin.products.edit', compact('product', 'planOptions', 'categories')); 
    }

    public function update(Request $request, Product $product)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:255|unique:products,code,' . $product->id,
            'category_id' => 'nullable|exists:categories,id', 
            'price' => 'required|numeric|min:0', 
            'discount_price' => 'nullable|numeric|min:0|lt:price', 
            'stock' => 'required|integer',
            'status' => 'required|in:active,inactive',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            
            // ✅✅✅ این بخش تغییر کرد ✅✅✅
            'plan_options' => 'nullable|array',
            'plan_options.*' => 'exists:plan_options,id',
            
            // ✅✅✅ START: اعتبارسنجی ویژگی‌ها ✅✅✅
            'features' => 'nullable|array|max:10',
            'features.*.key' => 'required_with:features.*.value|string|max:255',
            'features.*.value' => 'required_with:features.*.key|string|max:255',
            // ✅✅✅ END: اعتبارسنجی ویژگی‌ها ✅✅✅
        ]);

        if ($request->hasFile('image')) {
            if ($product->image_path) {
                Storage::disk('public')->delete($product->image_path);
            }
            $path = $request->file('image')->store('products', 'public');
            $validatedData['image_path'] = $path;
        }

        $validatedData['discount_price'] = $validatedData['discount_price'] ?? null;
        
        // ✅✅✅ START: آماده‌سازی ویژگی‌ها برای ذخیره ✅✅✅
        $validatedData['features'] = $this->prepareFeatures($request->features);
        // ✅✅✅ END: آماده‌سازی ویژگی‌ها ✅✅✅

        $product->update($validatedData);
        
        // ✅✅✅ این بخش تغییر کرد ✅✅✅
        $product->planOptions()->sync($request->plan_options);

        return redirect()->route('admin.products.index');
    }

    public function destroy(Product $product)
    {
        // ... (این متد بدون تغییر باقی می‌ماند) ...
        if ($product->image_path) {
            Storage::disk('public')->delete($product->image_path);
        }
        $product->delete();
        
        return redirect()->route('admin.products.index');
    }
    
    // ✅✅✅ START: افزودن متد کمکی ✅✅✅
    /**
     * آرایه ویژگی‌ها را برای ذخیره در دیتابیس آماده می‌کند
     * (جفت‌های خالی را حذف می‌کند)
     */
    private function prepareFeatures(?array $features): ?array
    {
        if (empty($features)) {
            return null;
        }
        
        // فیلتر کردن ردیف‌هایی که کلید یا مقدار آن‌ها خالی است
        $filtered = array_filter($features, function($feature) {
            return !empty($feature['key']) && !empty($feature['value']);
        });
        
        // اطمینان از اینکه ایندکس‌ها از 0 شروع می‌شوند (برای ذخیره‌سازی صحیح JSON)
        return array_values($filtered);
    }
    // ✅✅✅ END: افزودن متد کمکی ✅✅✅
}