<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\User;
use App\Models\Product; // ✅ اضافه شد
use App\Models\Category; // ✅ اضافه شد
use Illuminate\Http\Request; 
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    /**
     * فاز ۱: نمایش ۱۰ نماینده برتر (صفحه اصلی گزارش)
     */
    public function index(Request $request) 
    {
        $from_date = $request->input('from_date');
        $to_date = $request->input('to_date');

        $query = Invoice::where('status', '!=', 'rejected');

        // اعمال فیلتر تاریخ
        if ($from_date) {
            $query->where('created_at', '>=', $from_date . ' 00:00:00');
        }
        if ($to_date) {
            $query->where('created_at', '<=', $to_date . ' 23:59:59');
        }

        $topAgents = $query->select('user_id', DB::raw('SUM(final_price) as total_sales'))
            ->groupBy('user_id')
            ->orderBy('total_sales', 'desc')
            ->with('user')
            ->take(10) 
            ->get();
            
        return view('admin.reports.index', compact('topAgents', 'from_date', 'to_date'));
    }

    /**
     * فاز ۲: نمایش همه نمایندگان با صفحه‌بندی
     */
    public function allAgents(Request $request) 
    {
        $from_date = $request->input('from_date');
        $to_date = $request->input('to_date');

        $query = Invoice::where('status', '!=', 'rejected');

        // اعمال فیلتر تاریخ
        if ($from_date) {
            $query->where('created_at', '>=', $from_date . ' 00:00:00');
        }
        if ($to_date) {
            $query->where('created_at', '<=', $to_date . ' 23:59:59');
        }
        
        $agents = $query->select('user_id', DB::raw('SUM(final_price) as total_sales'))
            ->groupBy('user_id')
            ->orderBy('total_sales', 'desc')
            ->with('user')
            ->paginate(50); 
            
        return view('admin.reports.all-agents', compact('agents', 'from_date', 'to_date'));
    }

    /**
     * فاز ۳: نمایش جزئیات عملکرد یک نماینده خاص
     */
    public function showAgent(Request $request, User $user) 
    {
        $from_date = $request->input('from_date');
        $to_date = $request->input('to_date');

        // ۱. کوئری پایه برای این نماینده
        $query = Invoice::where('user_id', $user->id);

        // اعمال فیلتر تاریخ به کوئری پایه
        if ($from_date) {
            $query->where('created_at', '>=', $from_date . ' 00:00:00');
        }
        if ($to_date) {
            $query->where('created_at', '<=', $to_date . ' 23:59:59');
        }

        // ۲. کپی منطق نمودار فروش ماهانه (از بخش ۴ داشبورد)
        $salesInvoicesQuery = $query->clone()->where('status', '!=', 'rejected');
        $salesInvoices = $salesInvoicesQuery->get();
        
        $salesByMonth = $salesInvoices->groupBy(fn($invoice) => verta($invoice->created_at)->format('F'))
                                    ->map(fn($month) => $month->sum('final_price'));
        
        $month_chart_labels = $salesByMonth->keys();
        $month_chart_data = $salesByMonth->values();

        // ۳. کپی منطق نمودار فروش بر اساس محصول (از بخش ۶ داشBورد)
        $productsQuery = DB::table('invoice_product')
            ->join('invoices', 'invoice_product.invoice_id', '=', 'invoices.id')
            ->join('products', 'invoice_product.product_id', '=', 'products.id')
            ->where('invoices.user_id', $user->id) 
            ->where('invoices.status', '!=', 'rejected');

        // اعمال فیلتر تاریخ به کوئری محصولات
        if ($from_date) {
            $productsQuery->where('invoices.created_at', '>=', $from_date . ' 00:00:00');
        }
        if ($to_date) {
            $productsQuery->where('invoices.created_at', '<=', $to_date . ' 23:59:59');
        }

        $productsData = $productsQuery->select('products.name', DB::raw('SUM(invoice_product.quantity) as total_quantity'))
            ->groupBy('products.name')
            ->orderBy('total_quantity', 'desc')
            ->take(10) 
            ->get();
            
        $products_chart_labels = $productsData->pluck('name');
        $products_chart_data = $productsData->pluck('total_quantity');

        // ۴. کپی منطق نمودار فروش بر اساس دسته‌بندی (از بخش ۶ داشبورد)
        $categoriesQuery = DB::table('invoice_product')
            ->join('invoices', 'invoice_product.invoice_id', '=', 'invoices.id')
            ->join('products', 'invoice_product.product_id', '=', 'products.id')
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->where('invoices.user_id', $user->id) 
            ->where('invoices.status', '!=', 'rejected');

        // اعمال فیلتر تاریخ به کوئری دسته‌بندی‌ها
        if ($from_date) {
            $categoriesQuery->where('invoices.created_at', '>=', $from_date . ' 00:00:00');
        }
        if ($to_date) {
            $categoriesQuery->where('invoices.created_at', '<=', $to_date . ' 23:59:59');
        }
            
        $categoriesData = $categoriesQuery->select('categories.name', DB::raw('COUNT(invoices.id) as total_sales_count'))
            ->groupBy('categories.name')
            ->orderBy('total_sales_count', 'desc')
            ->get();

        $categories_chart_labels = $categoriesData->pluck('name');
        $categories_chart_data = $categoriesData->pluck('total_sales_count');


        return view('admin.reports.agent-show', compact(
            'user', 
            'month_chart_labels', 
            'month_chart_data', 
            'products_chart_labels', 
            'products_chart_data', 
            'categories_chart_labels', 
            'categories_chart_data',
            'from_date',
            'to_date'
        ));
    }

    // =============================================
    // ## START: ✅ متدهای جدید گزارش‌ها ##
    // =============================================

    /**
     * گزارش محصولات پرفروش (سراسری)
     */
    public function topProducts(Request $request)
    {
        $from_date = $request->input('from_date');
        $to_date = $request->input('to_date');

        $query = DB::table('invoice_product')
            ->join('invoices', 'invoice_product.invoice_id', '=', 'invoices.id')
            ->join('products', 'invoice_product.product_id', '=', 'products.id')
            ->where('invoices.status', '!=', 'rejected');

        if ($from_date) {
            $query->where('invoices.created_at', '>=', $from_date . ' 00:00:00');
        }
        if ($to_date) {
            $query->where('invoices.created_at', '<=', $to_date . ' 23:59:59');
        }

        $products = $query->select(
                'products.name', 
                'products.code', 
                DB::raw('SUM(invoice_product.quantity) as total_quantity'),
                DB::raw('SUM(invoice_product.price * invoice_product.quantity) as total_amount')
            )
            ->groupBy('products.id', 'products.name', 'products.code')
            ->orderBy('total_amount', 'desc')
            ->paginate(50);

        return view('admin.reports.top-products', compact('products', 'from_date', 'to_date'));
    }

    /**
     * گزارش محصولات راکد
     */
    public function deadStock(Request $request)
    {
        $from_date = $request->input('from_date');
        $to_date = $request->input('to_date');

        $query = Product::where('stock', '>', 0);

        // اگر تاریخ وارد شده بود، فقط محصولاتی را پیدا کن که در آن بازه فروخته نشده‌اند
        if ($from_date || $to_date) {
            $query->whereDoesntHave('invoices', function ($q) use ($from_date, $to_date) {
                $q->where('invoices.status', '!=', 'rejected');
                if ($from_date) {
                    $q->where('invoices.created_at', '>=', $from_date . ' 00:00:00');
                }
                if ($to_date) {
                    $q->where('invoices.created_at', '<=', $to_date . ' 23:59:59');
                }
            });
        } else {
            // اگر تاریخی نبود، محصولاتی که کلا فروشی نداشته‌اند
            $query->whereDoesntHave('invoices');
        }

        $products = $query->select('name', 'code', 'stock')->paginate(50);

        return view('admin.reports.dead-stock', compact('products', 'from_date', 'to_date'));
    }

    /**
     * گزارش فروش بر اساس دسته‌بندی
     */
    public function byCategory(Request $request)
    {
        $from_date = $request->input('from_date');
        $to_date = $request->input('to_date');

        $query = DB::table('invoice_product')
            ->join('invoices', 'invoice_product.invoice_id', '=', 'invoices.id')
            ->join('products', 'invoice_product.product_id', '=', 'products.id')
            // از Left Join استفاده می‌کنیم تا محصولات "بدون دسته‌بندی" هم شمارش شوند
            ->leftJoin('categories', 'products.category_id', '=', 'categories.id')
            ->where('invoices.status', '!=', 'rejected');

        if ($from_date) {
            $query->where('invoices.created_at', '>=', $from_date . ' 00:00:00');
        }
        if ($to_date) {
            $query->where('invoices.created_at', '<=', $to_date . ' 23:59:59');
        }

        $categories = $query->select(
                'categories.name',
                DB::raw('COUNT(DISTINCT invoices.id) as total_invoices'),
                DB::raw('SUM(invoice_product.quantity) as total_quantity'),
                DB::raw('SUM(invoice_product.price * invoice_product.quantity) as total_amount')
            )
            ->groupBy('categories.id', 'categories.name')
            ->orderBy('total_amount', 'desc')
            ->paginate(50);
            
        return view('admin.reports.by-category', compact('categories', 'from_date', 'to_date'));
    }

    /**
     * گزارش محبوبیت گزینه‌های اقساط
     */
    public function planOptions(Request $request)
    {
        $from_date = $request->input('from_date');
        $to_date = $request->input('to_date');

        $query = DB::table('invoices')
            ->join('plan_options', 'invoices.plan_option_id', '=', 'plan_options.id')
            ->join('plans', 'plan_options.plan_id', '=', 'plans.id')
            ->where('invoices.status', '!=', 'rejected');
        
        if ($from_date) {
            $query->where('invoices.created_at', '>=', $from_date . ' 00:00:00');
        }
        if ($to_date) {
            $query->where('invoices.created_at', '<=', $to_date . ' 23:59:59');
        }

        $planOptions = $query->select(
                'plans.name as plan_name',
                'plans.type as plan_type',
                'plan_options.months',
                DB::raw('COUNT(invoices.id) as total_uses'),
                DB::raw('SUM(invoices.final_price) as total_sales')
            )
            ->groupBy('plan_options.id', 'plans.name', 'plans.type', 'plan_options.months')
            ->orderBy('total_uses', 'desc')
            ->paginate(50);
            
        return view('admin.reports.plan-options', compact('planOptions', 'from_date', 'to_date'));
    }

    // =============================================
    // ## END: ✅ متدهای جدید گزارش‌ها ##
    // =============================================
}