<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
// use App\Models\UserDocument; // (نیازی نیست، از طریق رابطه ذخیره می‌کنیم)
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB; 
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage; 
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules;
use App\Imports\UsersImport;
use Maatwebsite\Excel\Facades\Excel;
use Maatwebsite\Excel\Validators\ValidationException;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $usersQuery = User::query();
        if (auth()->user()->role == 'super_admin') {
            $usersQuery->where('id', '!=', auth()->id());
        } 
        else if (auth()->user()->role == 'regional_manager') {
            $usersQuery->where('creator_id', auth()->id());
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $usersQuery->where('status', $request->status);
        }

        $sortableColumns = ['name', 'email', 'created_at', 'status']; 
        $sort = in_array($request->query('sort'), $sortableColumns) ? $request->query('sort') : 'created_at';
        $direction = in_array($request->query('direction'), ['asc', 'desc']) ? $request->query('direction') : 'desc';
        
        $users = $usersQuery->orderBy($sort, $direction)->paginate(15)->appends($request->query());
        
        return view('admin.users.index', compact('users', 'sort', 'direction'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.users.create');
    }


    /**
     * Store a newly created resource in storage.
     * (✅✅✅ اصلاح شده نهایی: حذف اعتبارسنجی status و تنظیم پیش‌فرض)
     */
    public function store(Request $request)
    {
        // ✅ 1. اعتبارسنجی 'status' حذف شد
        $validatedData = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'mobile' => ['required', 'string', 'regex:/^09[0-9]{9}$/', 'unique:users'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'string', Rule::in(['regional_manager', 'agent', 'support', 'accountant'])],
            // 'status' => ..., // <-- حذف شد
            'documents' => ['nullable', 'array', 'max:8'],
            'documents.*.file' => ['nullable', 'file', 'mimes:jpg,jpeg,png,pdf', 'max:2048'],
            'documents.*.title' => ['nullable', 'string', 'max:255'],
        ]);

        // =============================================
        // ## START: (فاز ۳) منطق جدید برای تعیین وضعیت ##
        // =============================================
        
        // ✅ 2. همه کاربران جدید "غیرفعال" ساخته می‌شوند.
        // فعال‌سازی فقط از طریق داشبورد (toggleStatus) توسط ادمین/حسابدار انجام می‌شود.
        $statusToSet = 'inactive';
        
        // (اگر می‌خواهید ادمین کل بتواند کاربر فعال بسازد، کد زیر را جایگزین خط بالا کنید)
        // $statusToSet = (auth()->user()->role === 'super_admin') ? 'active' : 'inactive';

        // =============================================
        // ## END: منطق جدید ##
        // =============================================

        DB::beginTransaction();
        try {
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'mobile' => $request->mobile,
                'password' => Hash::make($request->password),
                'role' => $request->role,
                'status' => $statusToSet, // ✅ 3. استفاده از متغیر جدید
                'creator_id' => auth()->id(),
            ]);

            if ($request->hasFile('documents')) {
                foreach ($request->file('documents') as $index => $fileData) {
                    if (empty($fileData['file'])) { continue; }
                    $path = $fileData['file']->store('user_documents/' . $user->id, 'public');
                    $title = $request->input('documents')[$index]['title'] ?? null;
                    $user->documents()->create(['title' => $title, 'file_path' => $path]);
                }
            }
            DB::commit(); 

            // =============================================
            // ## START: (فاز ۱) اصلاح ریدایرکت داینامیک ##
            // =============================================
            $redirectRoute = 'admin.users.index'; // پیش‌فرض برای سوپر ادمین
            if (auth()->user()->role === 'regional_manager') {
                $redirectRoute = 'regional-manager.users.index';
            }
            
            return redirect()->route($redirectRoute)->with('success', 'کاربر با موفقیت ایجاد شد (وضعیت: غیرفعال).');
            // =============================================
            // ## END: اصلاح ریدایرکت ##
            // =============================================

        } catch (\Exception $e) {
            DB::rollBack(); 
            return back()->with('error', 'خطا در ایجاد کاربر: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        return view('admin.users.edit', compact('user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'mobile' => ['required', 'string', 'regex:/^09[0-9]{9}$/', Rule::unique('users')->ignore($user->id)],
            'role' => ['required', 'string', Rule::in(['regional_manager', 'agent', 'support', 'accountant'])],
            'status' => ['required', 'string', Rule::in(['active', 'inactive'])], 
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
        ]);
        
        $userData = $request->only('name', 'email', 'mobile', 'role', 'status');
        if ($request->filled('password')) {
            $userData['password'] = Hash::make($request->password);
        }
        $user->update($userData);
        return redirect()->route('admin.users.index')->with('success', 'کاربر با موفقیت به‌روزرسانی شد.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        Storage::disk('public')->deleteDirectory('user_documents/' . $user->id);
        $user->delete(); 
        return redirect()->route('admin.users.index')->with('success', 'کاربر با موفقیت حذف شد.');
    }

    /**
     * Show the form for importing users.
     */
    public function showImportForm()
    {
        $this->authorize('is-super-admin');
        return view('admin.users.import');
    }
    
    /**
     * Handle the import of users.
     */
    public function import(Request $request)
    {
        // ... (کد ایمپورت)
    }

    /**
     * فعال/غیرفعال کردن وضعیت کاربر (برای داشبورد)
     */
    public function toggleStatus(Request $request, User $user)
    {
        // فقط ادمین کل یا حسابدار می‌توانند وضعیت را تغییر دهند
        if (auth()->user()->role !== 'super_admin' && auth()->user()->role !== 'accountant') {
            abort(403);
        }

        // فقط نماینده‌ها قابل تغییر هستند
        if ($user->role !== 'agent') {
            return response()->json(['error' => 'فقط وضعیت نمایندگان قابل تغییر است.'], 403);
        }

        try {
            $newStatus = $user->status === 'active' ? 'inactive' : 'active';
            $user->update(['status' => $newStatus]);
            
            return response()->json([
                'success' => true, 
                'new_status' => $newStatus,
                'message' => 'وضعیت نماینده با موفقیت به ' . ($newStatus === 'active' ? 'فعال' : 'غیر فعال') . ' تغییر یافت.'
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'خطا در تغییر وضعیت: ' . $e->getMessage()], 500);
        }
    }
}