<?php

namespace App\Http\Controllers\Agent;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class CustomerController extends Controller
{
    /**
     * Display a listing of the resource.
     * (بدون تغییر)
     */
    public function index()
    {
        $customers = auth()->user()->customers()->latest()->paginate(15);
        return view('agent.customers.index', compact('customers'));
    }

    /**
     * Show the form for creating a new resource.
     * (بدون تغییر)
     */
    public function create()
    {
        return view('agent.customers.create');
    }

    /**
     * Store a newly created resource in storage.
     * (✅ description اضافه شد)
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'family_name' => ['required', 'string', 'max:255'],
            'national_id' => ['nullable', 'string', 'max:255', Rule::unique('customers')->where(fn ($query) => $query->where('agent_id', auth()->id()))],
            'mobile' => ['nullable', 'string', 'max:255', Rule::unique('customers')->where(fn ($query) => $query->where('agent_id', auth()->id()))],
            'phone' => ['nullable', 'string', 'max:255'],
            'province' => ['nullable', 'string', 'max:255'],
            'city' => ['nullable', 'string', 'max:255'],
            'postal_code' => ['nullable', 'string', 'max:255'],
            'address' => ['nullable', 'string'],
            'description' => ['nullable', 'string'], // ✅ اعتبارسنجی توضیحات
            'birth_date_day' => ['nullable', 'integer', 'min:1', 'max:31'],
            'birth_date_month' => ['nullable', 'integer', 'min:1', 'max:12'],
            'birth_date_year' => ['nullable', 'integer', 'min:1300', 'max:1450'],
        ]);

        // ترکیب نام و نام خانوادگی
        $validatedData['name'] = $validatedData['name'] . ' ' . $validatedData['family_name'];
        unset($validatedData['family_name']);

        // ترکیب تاریخ تولد
        if ($request->filled('birth_date_year') && $request->filled('birth_date_month') && $request->filled('birth_date_day')) {
            $validatedData['birth_date'] = $validatedData['birth_date_year'] . '-' . $validatedData['birth_date_month'] . '-' . $validatedData['birth_date_day'];
        }
        unset($validatedData['birth_date_day'], $validatedData['birth_date_month'], $validatedData['birth_date_year']);

        auth()->user()->customers()->create($validatedData);

        return redirect()->route('agent.customers.index')->with('success', 'مشتری با موفقیت ایجاد شد.');
    }

    /**
     * Show the form for editing the specified resource.
     * (منطق تفکیک نام و تاریخ تولد بدون تغییر)
     */
    public function edit(Customer $customer)
    {
        if ($customer->agent_id !== auth()->id()) {
            abort(403);
        }

        // --- منطق تفکیک برای نمایش در فرم ---
        // تفکیک نام به نام و نام خانوادگی (بر اساس اولین فاصله)
        $nameParts = explode(' ', $customer->name, 2);
        $customer->first_name = $nameParts[0] ?? '';
        $customer->family_name = $nameParts[1] ?? '';

        // تفکیک تاریخ تولد
        $birthDateParts = $customer->birth_date ? explode('-', $customer->birth_date) : [null, null, null];
        $customer->birth_date_year = $birthDateParts[0];
        $customer->birth_date_month = $birthDateParts[1];
        $customer->birth_date_day = $birthDateParts[2];
        // --- پایان منطق تفکیک ---

        return view('agent.customers.edit', compact('customer'));
    }

    /**
     * Update the specified resource in storage.
     * (✅✅✅ متد بازنویسی شد تا فقط توضیحات را آپدیت کند ✅✅✅)
     */
    public function update(Request $request, Customer $customer)
    {
        if ($customer->agent_id !== auth()->id()) {
            abort(403);
        }

        // اعتبارسنجی فقط برای فیلد توضیحات
        $validatedData = $request->validate([
            'description' => ['nullable', 'string'],
        ]);

        $customer->update($validatedData);

        return redirect()->route('agent.customers.index')->with('success', 'توضیحات مشتری با موفقیت به‌روزرسانی شد.');
    }

    /**
     * Remove the specified resource from storage.
     * (بدون تغییر)
     */
    public function destroy(Customer $customer)
    {
        if ($customer->agent_id !== auth()->id()) {
            abort(403);
        }

        // (توجه: اگر مشتری فاکتور داشته باشد، باید منطق حذف فاکتورها نیز بررسی شود)
        $customer->delete();
        return back()->with('success', 'مشتری با موفقیت حذف شد.');
    }
}