<?php

namespace App\Http\Controllers\Agent;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use App\Models\Product;
use App\Models\PlanOption;
use App\Models\Invoice;
use App\Models\Setting;
use App\Models\Bundle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Session; 

class InvoiceController extends Controller
{
    /**
     * قدم اول: نمایش لیست طرح‌ها
     * (✅✅✅ اصلاح شد: فقط طرح‌هایی را نشان می‌دهد که گزینه‌های آنها محصول دارند)
     */
    public function listPlans()
    {
        // منطق قبلی:
        // $plans = Plan::whereHas('products')->latest()->get();
        
        // ✅ منطق جدید:
        // فقط طرح‌هایی را نشان بده که گزینه‌هایشان (options) به محصول متصل هستند
        $plans = Plan::whereHas('options.products')->latest()->get();
        
        return view('agent.plans-list', compact('plans'));
    }

    /**
     * قدم دوم: نمایش گزینه‌های یک طرح
     * (✅✅✅ اصلاح شد: فقط گزینه‌هایی را نشان می‌دهد که محصول دارند)
     */
    public function listPlanOptions(Plan $plan)
    {
        // منطق قبلی:
        // $plan->load('options');
        
        // ✅ منطق جدید:
        // فقط گزینه‌هایی را بارگذاری کن که به محصول متصل هستند
        $plan->load(['options' => fn($q) => $q->whereHas('products')]);
        
        return view('agent.plan-options', compact('plan')); 
    }

    /**
     * قدم سوم: ایجاد سبد خرید و نمایش لیست محصولات
     * (منطق این متد بر اساس گزینه طرح بود و صحیح است - بدون تغییر)
     */
    public function startCart(PlanOption $option)
    {
        // ایجاد یک سبد خرید جدید در سشن
        $cart = [
            'plan_option_id' => $option->id,
            'plan_option_details' => [ 
                'plan_id' => $option->plan_id, // (این دیگر برای فیلتر نیاز نیست، اما برای نمایش خوب است)
                'name' => $option->plan->name,
                'months' => $option->months,
                'coefficient' => $option->coefficient,
            ],
            'items' => [], 
            'totals' => [
                'base_price' => 0,
                'final_price' => 0,
            ]
        ];
        
        Session::put('invoice_cart', $cart);

        // هدایت به صفحه لیست محصولات
        return redirect()->route('agent.cart.products');
    }

    /**
     * قدم چهارم: نمایش لیست محصولات و باندل‌ها
     * (✅✅✅ اصلاح شد: محصولات بر اساس "گزینه طرح" فیلتر می‌شوند)
     */
    public function showProductList(Request $request)
    {
        $cart = Session::get('invoice_cart');
        
        if (!$cart) {
            return redirect()->route('agent.plans.list');
        }

        // $planId = $cart['plan_option_details']['plan_id']; // ❌ منطق قدیمی
        
        // ✅✅✅ منطق جدید ✅✅✅
        $planOptionId = $cart['plan_option_id']; // واکشی شناسه گزینه (مثلاً ۱۲ ماهه)

        // ۱. واکشی محصولات
        $queryProducts = Product::where('status', 'active')
                         // فیلتر محصولاتی که به این "گزینه" خاص متصل هستند
                         ->whereHas('planOptions', fn($q) => $q->where('plan_option_id', $planOptionId));

        if ($request->filled('search')) {
            $queryProducts->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('code', 'like', '%' . $request->search . '%');
            });
        }
        $products = $queryProducts->paginate(20, ['*'], 'products_page'); // صفحه‌بندی مجزا

        // ۲. واکشی باندل‌ها (منطق باندل‌ها فعلا به طرح اصلی متصل نیست، پس بدون تغییر)
        $queryBundles = Bundle::where('status', true);
        if ($request->filled('search')) {
            $queryBundles->where('name', 'like', '%' . $request->search . '%');
        }
        $bundles = $queryBundles->with('products')->paginate(10, ['*'], 'bundles_page'); // صفحه‌بندی مجزا

        $planOptionDetails = $cart['plan_option_details'];

        return view('agent.cart-products', compact('products', 'bundles', 'planOptionDetails', 'cart'));
    }

    /**
     * (عملیات) افزودن/آپدیت آیتم (محصول یا باندل) در سبد خرید
     * (✅✅✅ اصلاح شد: بررسی می‌کند آیا محصول به این گزینه طرح متصل است یا خیر)
     */
    public function cartAdd(Request $request)
    {
        $validated = $request->validate([
            'item_id' => ['required', 'integer', 'min:1'],
            'item_type' => ['required', Rule::in(['product', 'bundle'])],
            'quantity' => ['required', 'integer', 'min:1'],
        ]);
        
        if (!Session::has('invoice_cart')) {
            return back()->with('error', 'سبد خرید یافت نشد.');
        }

        $cart = Session::get('invoice_cart');
        $itemId = $validated['item_id'];
        $itemType = $validated['item_type'];
        $quantity = $validated['quantity'];
        
        $cartItemId = $itemType . '_' . $itemId; 

        try {
            if ($itemType === 'product') {
                
                // ✅✅✅ اعتبارسنجی اتصال محصول به گزینه طرح ✅✅✅
                $planOptionId = $cart['plan_option_id'];
                $product = Product::where('id', $itemId)
                                  // بررسی می‌کند آیا این محصول به گزینه طرح موجود در سبد خرید متصل است
                                  ->whereHas('planOptions', fn($q) => $q->where('plan_option_id', $planOptionId))
                                  ->first();

                if (!$product) {
                    throw new \Exception('این محصول برای طرح انتخاب شده مجاز نیست.');
                }
                // ✅✅✅ پایان اعتبارسنجی ✅✅✅

                if ($product->stock < $quantity) {
                    throw new \Exception('موجودی انبار محصول ' . $product->name . ' کافی نیست.');
                }
                
                // افزودن محصول به سبد
                $cart['items'][$cartItemId] = [
                    'item_id' => $product->id,
                    'type' => 'product',
                    'name' => $product->name,
                    'code' => $product->code,
                    'quantity' => $quantity,
                    'price' => $product->active_price, // قیمت پایه (یا تخفیفی)
                ];

            } elseif ($itemType === 'bundle') {
                // (منطق باندل‌ها بدون تغییر باقی می‌ماند)
                $bundle = Bundle::with('products')->find($itemId);
                
                foreach ($bundle->products as $product) {
                    $requiredStock = $product->pivot->quantity * $quantity;
                    if ($product->stock < $requiredStock) {
                        throw new \Exception('موجودی انبار محصول ' . $product->name . ' (داخل باندل) کافی نیست.');
                    }
                }
                
                $cart['items'][$cartItemId] = [
                    'item_id' => $bundle->id,
                    'type' => 'bundle',
                    'name' => $bundle->name,
                    'code' => 'B-' . $bundle->id,
                    'quantity' => $quantity,
                    'price' => $bundle->price, // قیمت خود باندل
                ];
            }

            // محاسبه مجدد جمع کل
            $this->recalculateCart($cart);
            Session::put('invoice_cart', $cart);

            return back()->with('success', 'آیتم به سبد اضافه شد.');

        } catch (\Exception $e) {
            return back()->with('error', 'خطا: ' . $e->getMessage());
        }
    }
    
    /**
     * (عملیات) حذف آیتم از سبد خرید
     * (بدون تغییر)
     */
    public function cartRemove(Request $request)
    {
        $request->validate(['cart_item_id' => 'required|string']);

        $cart = Session::get('invoice_cart');
        
        if (isset($cart['items'][$request->cart_item_id])) {
            unset($cart['items'][$request->cart_item_id]);
            $this->recalculateCart($cart);
            Session::put('invoice_cart', $cart);
        }
        return back()->with('success', 'آیتم از سبد حذف شد.');
    }

    /**
     * (عملیات) تابع کمکی برای محاسبه جمع کل سبد
     * (بدون تغییر)
     */
    private function recalculateCart(&$cart) // (pass by reference)
    {
        $totalBase = 0;
        
        foreach ($cart['items'] as $item) {
            $totalBase += $item['price'] * $item['quantity'];
        }
        
        // محاسبه قیمت نهایی بر اساس ضریب طرح
        $option = $cart['plan_option_details'];
        $finalPrice = $totalBase * (1 + ($option['months'] * $option['coefficient'] / 100));

        $cart['totals']['base_price'] = $totalBase;
        $cart['totals']['final_price'] = round($finalPrice); // گرد کردن قیمت نهایی
    }


    /**
     * قدم پنجم: نمایش سبد خرید (پیش‌نمایش نهایی)
     * (بدون تغییر)
     */
    public function showCart()
    {
        $cart = Session::get('invoice_cart');
        if (!$cart || empty($cart['items'])) {
            return redirect()->route('agent.plans.list')->with('error', 'سبد خرید شما خالی است.');
        }
        
        $customers = auth()->user()->customers()->select('id', 'name', 'national_id')->get();
        return view('agent.cart-review', compact('cart', 'customers')); 
    }


    /**
     * قدم نهایی: ذخیره پیش فاکتور (سبد خرید) در دیتابیس
     * (بدون تغییر)
     */
    public function storeInvoice(Request $request)
    {
        $cart = Session::get('invoice_cart');
        if (!$cart) return redirect()->route('agent.plans.list');

        $validatedData = $request->validate([
            'customer_id' => ['required', 'exists:customers,id', Rule::exists('customers', 'id')->where('agent_id', auth()->id())],
            'documents.*' => ['nullable', 'file', 'mimes:jpg,jpeg,png,pdf', 'max:2048'],
        ]);

        $documentPaths = [];
        if ($request->hasFile('documents')) {
            foreach ($request->file('documents') as $file) {
                $documentPaths[] = $file->store('documents', 'public');
            }
        }

        try {
            DB::beginTransaction();

            // ۱. ساخت پیش فاکتور اصلی
            $invoice = auth()->user()->invoices()->create([
                'plan_option_id' => $cart['plan_option_id'],
                'customer_id' => $validatedData['customer_id'],
                'final_price' => $cart['totals']['final_price'],
                'paid_amount_cash' => 0,
                'document_paths' => json_encode($documentPaths),
                'status' => 'pending',
            ]);

            // ۲. بررسی موجودی و اتصال آیتم‌ها (ذخیره سبد خرید)
            foreach ($cart['items'] as $item) {
                
                if ($item['type'] === 'product') {
                    // --- منطق محصول تکی ---
                    $product = Product::lockForUpdate()->find($item['item_id']);
                    if ($product->stock < $item['quantity']) {
                        throw new \Exception('موجودی محصول ' . $product->name . ' کافی نیست.');
                    }
                    
                    // کاهش موجودی
                    $product->decrement('stock', $item['quantity']);

                    // ذخیره در جدول میانی invoice_product
                    $invoice->products()->attach($item['item_id'], [
                        'quantity' => $item['quantity'],
                        'price' => $item['price'] // قیمت پایه در لحظه خرید
                    ]);

                } elseif ($item['type'] === 'bundle') {
                    // --- منطق باندل (Explode) ---
                    $bundle = Bundle::with('products')->find($item['item_id']);
                    
                    foreach ($bundle->products as $product) {
                        $productInStock = Product::lockForUpdate()->find($product->id);
                        $requiredStock = $product->pivot->quantity * $item['quantity'];
                        
                        if ($productInStock->stock < $requiredStock) {
                            throw new \Exception('موجودی محصول ' . $productInStock->name . ' (داخل باندل) کافی نیست.');
                        }
                        
                        // کاهش موجودی
                        $productInStock->decrement('stock', $requiredStock);

                        // ذخیره *محصول تشکیل‌دهنده* در جدول میانی
                        $invoice->products()->attach($productInStock->id, [
                            'quantity' => $requiredStock,
                            'price' => $productInStock->active_price // قیمت پایه خود محصول
                        ]);
                    }
                }
            }

            DB::commit();

            // ۴. پاک کردن سبد خرید از سشن
            Session::forget('invoice_cart');
            
            // ۵. هدایت به پرداخت
            return redirect()->route('payment.start', $invoice->id)
                             ->with('success', 'پیش فاکتور با موفقیت ایجاد شد. لطفاً پرداخت را تکمیل کنید.');

        } catch (\Exception $e) {
            DB::rollBack();
            foreach ($documentPaths as $path) {
                Storage::disk('public')->delete($path);
            }
            return back()->with('error', 'خطا در ثبت پیش فاکتور: ' . $e->getMessage());
        }
    }

    
    //
    // --- متدهای لیست پیش فاکتورها و پرینت (بدون تغییر) ---
    //
    
    public function index(Request $request)
    {
        $query = auth()->user()->invoices()->with([
            'customer', 
            'planOption.plan'
        ])->withCount('products'); 

        // ... (فیلترها) ...
        
        $invoices = $query->latest()->paginate(9)->appends($request->query());

        return view('agent.invoices.index', compact('invoices'));
    }
    
    public function printInvoice(Invoice $invoice)
        {
            if ($invoice->user_id !== auth()->id()) abort(403);
            
            $invoice->load('customer', 'planOption.plan', 'products'); 
            $settings = Setting::pluck('value', 'key')->all();
            
            // ✅✅✅ تغییر اینجاست
            return view('agent.show-invoice', [ 
                'invoice' => $invoice,
                'settings' => $settings
            ]);
        }
}