<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Product;
use App\Models\User;
use App\Models\Category; 
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        $dashboardData = [];
        $query = null;

        // ... (بخش ۱: تعیین کوئری پایه بر اساس نقش - بدون تغییر) ...
        if ($user->role === 'agent') {
            $query = Invoice::where('user_id', $user->id);
        } elseif ($user->role === 'regional_manager') {
            $agentIds = User::where('creator_id', $user->id)->pluck('id');
            $query = Invoice::whereIn('user_id', $agentIds);
        } elseif ($user->role === 'super_admin' || $user->role === 'accountant') { 
            $query = Invoice::query();
        }

        // ... (بخش ۲: فیلتر بر اساس نماینده - بدون تغییر) ...
        $selectedAgentId = null;
        if (($user->role === 'super_admin' || $user->role === 'accountant') && $request->filled('agent_id')) {
            $selectedAgentId = $request->agent_id;
            $query->where('user_id', $selectedAgentId);
            $dashboardData['selected_agent'] = User::find($selectedAgentId);
        }
        if ($user->role === 'super_admin' || $user->role === 'accountant') {
            $dashboardData['agents_list'] = User::where('role', 'agent')->select('id', 'name', 'status')->get();
        }

        // ==================================================
        // ۳. ✅✅✅ محاسبه آمارها (باکس‌های بالای صفحه) - اصلاح شد ✅✅✅
        // ==================================================
        $statsQuery = $query->clone(); 
        $stats = $statsQuery->select(
                'status', 
                DB::raw('COUNT(*) as count'), 
                DB::raw('SUM(final_price) as total_amount')
            )
            ->groupBy('status')
            ->get()
            ->keyBy('status')
            ->toArray();

        // ✅ آرایه وضعیت‌ها با وضعیت‌های جدید به‌روز شد
        $statuses = [
            'pending', 
            'processing', 
            'ready_to_ship', // جدید
            'shipped', 
            'delivered', 
            'returned',      // جدید
            'rejected'
        ];
        
        $formattedStats = [];
        foreach ($statuses as $status) {
            $formattedStats[$status] = [
                'count' => $stats[$status]['count'] ?? 0,
                'amount' => $stats[$status]['total_amount'] ?? 0,
            ];
        }
        $dashboardData['stats'] = $formattedStats;
        
        
        // ==================================================
        // ۴. ✅✅✅ داده‌های نمودارها - اصلاح شد ✅✅✅
        // ==================================================
        $salesInvoicesQuery = $query->clone()->where('status', '!=', 'rejected');
        $salesInvoices = $salesInvoicesQuery->get();

        // ✅ نمودار دایره‌ای وضعیت‌ها (فقط وضعیت‌های پس از پرداخت)
        $postPaymentStatuses = ['processing', 'ready_to_ship', 'shipped', 'delivered', 'returned'];
        $pieChartData = $salesInvoices
            ->whereIn('status', $postPaymentStatuses) // فیلتر فقط وضعیت‌های پردازش
            ->groupBy('status')
            ->map->count();

        // (مطمئن شوید این‌ها را ترجمه کرده‌اید)
        $dashboardData['status_chart_labels'] = $pieChartData->keys()->map(fn($status) => __($status));
        $dashboardData['status_chart_data'] = $pieChartData->values();

        // ✅ نمودار فروش ماهانه (۳ ماه گذشته شمسی)
        $salesByMonth = $salesInvoices
            ->groupBy(fn($invoice) => verta($invoice->created_at)->format('Y-m')) // Group by YYYY-MM
            ->map(fn($month) => $month->sum('final_price'));

        // ایجاد آرایه ۳ ماه گذشته (شمسی)
        $months = collect([
            verta()->subMonths(2),
            verta()->subMonths(1),
            verta() // ماه جاری
        ]);

        $month_chart_labels = $months->map(fn($date) => $date->format('F')); // [ 'شهریور', 'مهر', 'آبان' ]
        $month_chart_data = $months->map(function($date) use ($salesByMonth) {
            $key = $date->format('Y-m');
            return $salesByMonth->get($key, 0); // دریافت فروش، یا 0 اگر فروشی نبود
        });
        
        $dashboardData['month_chart_labels'] = $month_chart_labels;
        $dashboardData['month_chart_data'] = $month_chart_data;
        
        // ==================================================
        // ۵. (ادمین) رتبه‌بندی‌ها (بدون تغییر)
        // ==================================================
        if ($user->role === 'super_admin') {
             $dashboardData['agent_rankings'] = Invoice::where('status', '!=', 'rejected')
                ->select('user_id', DB::raw('SUM(final_price) as total_sales'))
                ->groupBy('user_id')->orderBy('total_sales', 'desc')->with('user')->take(5)->get();
        }

        // ==================================================
        // ۶. (ادمین) نمودارهای جدید فاز ۴ (بدون تغییر)
        // ==================================================
        if (($user->role === 'super_admin' || $user->role === 'accountant') && $selectedAgentId) 
        {
            // ... (منطق نمودار محصولات و دسته‌بندی‌ها بدون تغییر) ...
            $productsData = DB::table('invoice_product')
                ->join('invoices', 'invoice_product.invoice_id', '=', 'invoices.id')
                ->join('products', 'invoice_product.product_id', '=', 'products.id')
                ->where('invoices.user_id', $selectedAgentId)
                ->where('invoices.status', '!=', 'rejected')
                ->select('products.name', DB::raw('SUM(invoice_product.quantity) as total_quantity'))
                ->groupBy('products.name')
                ->orderBy('total_quantity', 'desc')
                ->take(10) 
                ->get();
            $dashboardData['products_chart_labels'] = $productsData->pluck('name');
            $dashboardData['products_chart_data'] = $productsData->pluck('total_quantity');
            $categoriesData = DB::table('invoice_product')
                ->join('invoices', 'invoice_product.invoice_id', '=', 'invoices.id')
                ->join('products', 'invoice_product.product_id', '=', 'products.id')
                ->join('categories', 'products.category_id', '=', 'categories.id')
                ->where('invoices.user_id', $selectedAgentId)
                ->where('invoices.status', '!=', 'rejected')
                ->select('categories.name', DB::raw('COUNT(invoices.id) as total_sales_count'))
                ->groupBy('categories.name')
                ->orderBy('total_sales_count', 'desc')
                ->get();
            $dashboardData['categories_chart_labels'] = $categoriesData->pluck('name');
            $dashboardData['categories_chart_data'] = $categoriesData->pluck('total_sales_count');
        }

        return view('dashboard', ['data' => $dashboardData]);
    }
}