<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage; 
use Illuminate\Validation\Rule; 

class PaymentController extends Controller
{
    /**
     * قدم ۱: نمایش صفحه شروع پرداخت
     */
    public function start(Invoice $invoice)
    {
        if ($invoice->user_id !== Auth::id()) abort(403);
        if ($invoice->status !== 'pending') {
            return redirect()->route('agent.invoices.index')->with('error', 'این پیش فاکتور قبلاً نهایی شده است.');
        }
        $invoice->load('customer', 'planOption.plan', 'products');
        return view('payment.start', compact('invoice'));
    }

    // ==========================================================
    // ## بخش پرداخت نقدی (منطق ریال) ##
    // (✅✅✅ اصلاح شده برای اختیاری بودن پرداخت نقدی و حجم فایل)
    // ==========================================================

    public function redirectToCashGateway(Request $request, Invoice $invoice)
    {
        if ($invoice->user_id !== Auth::id()) abort(403);
        
        $remainingAmountRial = ($invoice->final_price - $invoice->paid_amount_cash);

        // --- ۱. اعتبارسنجی (اصلاح شده) ---
        $validationRules = [
            // ✅ "required" حذف شد و "min" به 0 تغییر کرد
            'partial_amount' => 'nullable|numeric|min:0|max:' . $remainingAmountRial,
        ];

        // اگر طرح چکی بود، اعتبارسنجی فایل‌ها را اضافه کن
        if ($invoice->planOption->plan->type === 'check') {
            $validationRules['cheques'] = ['required', 'array', 'min:1', 'max:16']; // حداقل 1 فایل، حداکثر 16
            // ✅ اعتبارسنجی حجم فایل (1024 کیلوبایت = 1 مگابایت) اضافه شد
            $validationRules['cheques.*'] = ['file', 'mimes:jpg,jpeg,png,pdf', 'max:1024']; 
        }
        
        $validated = $request->validate($validationRules);
        
        // --- ۲. ذخیره فایل‌های چک (بدون تغییر) ---
        if ($invoice->planOption->plan->type === 'check') {
            $chequePaths = [];
            // ابتدا فایل‌های قبلی (اگر وجود داشت) را پاک کن
            if ($invoice->cheque_document_paths) {
                $oldPaths = json_decode($invoice->cheque_document_paths, true);
                if (is_array($oldPaths)) {
                    foreach ($oldPaths as $oldPath) {
                        Storage::disk('public')->delete($oldPath);
                    }
                }
            }

            // ذخیره فایل‌های جدید
            foreach ($request->file('cheques') as $file) {
                $chequePaths[] = $file->store('cheques/' . $invoice->id, 'public');
            }
            
            // ذخیره آرایه مسیرها در دیتابیس
            $invoice->update(['cheque_document_paths' => json_encode($chequePaths)]);
        }

        // --- ۳. منطق پرداخت نقدی (اصلاح شده) ---
        
        // ✅ اگر مبلغی وارد نشده بود، آن را صفر در نظر بگیر
        $amountToPayRial = $validated['partial_amount'] ?? 0;
        
        // ✅ اگر مبلغ پرداخت صفر بود، به درگاه نرو و مستقیم به صفحه پرداخت برگرد
        if ($amountToPayRial <= 0) {
            // اگر طرح چکی بود، یعنی مدارک آپلود شده‌اند
            if ($invoice->planOption->plan->type === 'check') {
                 return redirect()->route('payment.start', $invoice->id)
                           ->with('success', 'مدارک چک با موفقیت آپلود شد. فاکتور در انتظار تایید حسابداری قرار گرفت.');
            }
            return back()->with('error', 'مبلغی برای پرداخت نقدی وارد نشد.');
        }

        // (اگر مبلغ بزرگتر از صفر بود، ادامه فرآیند درگاه)
        session(['cash_payment_amount' => $amountToPayRial]);

        // (شبیه‌سازی درگاه سامان)
        return redirect()->route('payment.cash.callback', [
            'invoice_id' => $invoice->id,
            'status' => 'success',
            'track_id' => 'SAMAN_SIM_' . time()
        ]);
    }

    /**
     * (متد handleCashCallback بدون تغییر باقی می‌ماند)
     */
    public function handleCashCallback(Request $request)
    {
        $request->validate([
            'invoice_id' => 'required|exists:invoices,id',
            'status' => 'required|string|in:success,failure',
            'track_id' => 'required|string',
        ]);
        
        $invoice = Invoice::findOrFail($request->invoice_id);

        $amountPaidRial = session()->pull('cash_payment_amount', 0);

        if ($request->status === 'success' && $amountPaidRial > 0) {
            
            $invoice->increment('paid_amount_cash', $amountPaidRial);

            if ($invoice->paid_amount_cash >= $invoice->final_price) {
                $invoice->update(['status' => 'processing']);
                return redirect()->route('agent.invoices.index')->with('success', 'کل مبلغ پیش فاکتور با موفقیت پرداخت و نهایی شد.');
            }

            return redirect()->route('payment.start', $invoice->id)
                           ->with('success', 'مبلغ ' . number_format($amountPaidRial) . ' ریال با موفقیت پرداخت شد.');
        
        } else {
            return redirect()->route('payment.start', $invoice->id)
                           ->with('error', 'پرداخت نقدی ناموفق بود.');
        }
    }


    // ==========================================================
    // ## بخش پرداخت اعتباری (منطق ریال) ##
    // (بدون تغییر)
    // ==========================================================
    public function redirectToGateway(Request $request, Invoice $invoice)
    {
        if ($invoice->user_id !== Auth::id()) abort(403);
        
        $remainingAmountRial = $invoice->final_price - $invoice->paid_amount_cash;

        if ($remainingAmountRial <= 0) {
            return redirect()->route('payment.start', $invoice->id)->with('error', 'مبلغی برای پرداخت اعتباری باقی نمانده است.');
        }

        $simulatedStatus = rand(0, 1) ? 'success' : 'failure';
        
        return redirect()->route('payment.callback', [
            'invoice_id' => $invoice->id,
            'status' => $simulatedStatus,
            'track_id' => 'BAJET_SIM_' . time()
        ]);
    }

    public function handleCallback(Request $request)
    {
        $request->validate([
            'invoice_id' => 'required|exists:invoices,id',
            'status' => 'required|string|in:success,failure',
            'track_id' => 'required|string',
        ]);

        $invoice = Invoice::findOrFail($request->invoice_id);

        if ($request->status === 'success') {
            
            $invoice->update(['status' => 'processing']);

            return redirect()->route('agent.invoices.index')
                           ->with('success', 'پرداخت اعتباری پیش فاکتور ' . $invoice->id . ' موفقیت‌آمیز بود.');
        
        } else {
            return redirect()->route('payment.start', $invoice->id)
                           ->with('error', 'پرداخت اعتباری ناموفق بود.');
        }
    }
}